"""オペレータ - マテリアル操作"""
import bpy
from bpy.types import Operator
from ..core.materials import create_outline_material, create_toon_material, create_toon_material_from_original, insert_toon_nodes_to_material, remove_toon_nodes_from_material
from ..core.modifiers import apply_inverted_hull
from ..core.render_engine import apply_render_optimizations
def apply_toon_to_object(obj, props, outline_mat, shared_toon_mat):
    """単一のオブジェクトにトゥーン設定を適用"""
    if obj.type != 'MESH':
        return False, 0
    applied = False
    insert_count = 0
    if props.material_mode == 'INSERT':
        if len(obj.data.materials) > 0:
            for mat in obj.data.materials:
                if mat and insert_toon_nodes_to_material(mat, props):
                    insert_count += 1
            has_outline = any(m and m.name == "OutlineMaterial" for m in obj.data.materials)
            if not has_outline:
                obj.data.materials.append(outline_mat)
            if apply_inverted_hull(obj, props):
                applied = True
        else:
            obj.data.materials.append(shared_toon_mat)
            obj.data.materials.append(outline_mat)
            if apply_inverted_hull(obj, props):
                applied = True
    elif props.material_mode == 'PRESERVE':
        original_mat = obj.data.materials[0] if len(obj.data.materials) > 0 else None
        toon_mat = create_toon_material_from_original(original_mat, props, obj.name)
        obj.data.materials.clear()
        obj.data.materials.append(toon_mat)
        obj.data.materials.append(outline_mat)
        if apply_inverted_hull(obj, props):
            applied = True
    else:
        obj.data.materials.clear()
        obj.data.materials.append(shared_toon_mat)
        obj.data.materials.append(outline_mat)
        if apply_inverted_hull(obj, props):
            applied = True
    return applied, insert_count
class OMOSEN_OT_ApplyToSelected(Operator):
    """選択オブジェクトに適用"""
    bl_idname = "omosen.apply_to_selected"
    bl_label = "Apply to Selected"
    bl_description = "Apply Toon Shader to selected mesh objects"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        props = context.scene.omosen_props
        outline_mat = create_outline_material(props)
        if props.material_mode != 'SHARED' and outline_mat.users > 0:
            outline_mat = outline_mat.copy()
        shared_toon_mat = None
        if props.material_mode in ['SHARED', 'INSERT']:
            shared_toon_mat = create_toon_material(props)
        success_count = 0
        insert_count = 0
        for obj in context.selected_objects:
            applied, count = apply_toon_to_object(obj, props, outline_mat, shared_toon_mat)
            if applied:
                success_count += 1
            insert_count += count
        mode_names = {'SHARED': 'Shared Material', 'PRESERVE': 'Preserve Textures', 'INSERT': 'Insert Nodes'}
        mode_name = mode_names.get(props.material_mode, props.material_mode)
        if success_count > 0 or insert_count > 0:
            tr = bpy.app.translations.pgettext_iface
            msg = tr("Applied to {} objects").format(success_count)
            if insert_count > 0:
                msg += tr(" ({} materials converted)").format(insert_count)
            msg += f" ({tr(mode_name)})"
            self.report({'INFO'}, msg)
        else:
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("No applicable mesh objects selected"))
        return {'FINISHED'}
class OMOSEN_OT_ApplyToAll(Operator):
    """全オブジェクトに適用"""
    bl_idname = "omosen.apply_to_all"
    bl_label = "Apply to All"
    bl_description = "Apply Toon Shader to all mesh objects in the scene"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        props = context.scene.omosen_props
        outline_mat = create_outline_material(props)
        shared_toon_mat = None
        if props.material_mode in ['SHARED', 'INSERT']:
            shared_toon_mat = create_toon_material(props)
        success_count = 0
        insert_count = 0
        for obj in context.scene.objects:
            applied, count = apply_toon_to_object(obj, props, outline_mat, shared_toon_mat)
            if applied:
                success_count += 1
            insert_count += count
        mode_names = {'SHARED': 'Shared Material', 'PRESERVE': 'Preserve Textures', 'INSERT': 'Insert Nodes'}
        mode_name = mode_names.get(props.material_mode, props.material_mode)
        if success_count > 0 or insert_count > 0:
            tr = bpy.app.translations.pgettext_iface
            msg = tr("Applied to {} objects").format(success_count)
            if insert_count > 0:
                msg += tr(" ({} materials converted)").format(insert_count)
            msg += f" ({tr(mode_name)})"
            self.report({'INFO'}, msg)
        else:
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("No applicable mesh objects found"))
        return {'FINISHED'}
class OMOSEN_OT_RemoveFromSelected(Operator):
    """選択から削除"""
    bl_idname = "omosen.remove_from_selected"
    bl_label = "Remove from Selected"
    bl_description = "Remove Toon Shader from selected objects"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        count = 0
        mat_count = 0
        for obj in context.selected_objects:
            if obj.type != 'MESH':
                continue
            removed_from_obj = False
            for mod in list(obj.modifiers):
                if mod.type == 'SOLIDIFY' and mod.name.startswith("ToonOutline"):
                    obj.modifiers.remove(mod)
                    removed_from_obj = True
            for mat_slot in obj.material_slots:
                if mat_slot.material:
                    mat = mat_slot.material
                    if mat.name == "ToonMaterial" or mat.name.startswith("ToonMaterial_"):
                        mat_slot.material = None
                        mat_count += 1
                        removed_from_obj = True
                    else:
                        if remove_toon_nodes_from_material(mat):
                            mat_count += 1
                            removed_from_obj = True
            indices_to_remove = []
            for i, mat_slot in enumerate(obj.material_slots):
                if mat_slot.material and mat_slot.material.name == "OutlineMaterial":
                    indices_to_remove.append(i)
            for i in reversed(indices_to_remove):
                obj.data.materials.pop(index=i)
                removed_from_obj = True
            if removed_from_obj:
                count += 1
        if count > 0:
            tr = bpy.app.translations.pgettext_iface
            msg = tr("Removed from {} objects").format(count)
            if mat_count > 0:
                msg += tr(" ({} materials restored)").format(mat_count)
            self.report({'INFO'}, msg)
        else:
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("No target found to remove"))
        return {'FINISHED'}
class OMOSEN_OT_CreateOutlineVertexGroup(Operator):
    """制御用頂点グループを作成"""
    bl_idname = "omosen.create_outline_vertex_group"
    bl_label = "Create Control Vertex Group"
    bl_description = "Create a vertex group to control outline thickness"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        props = context.scene.omosen_props
        if not props.outline_vertex_group:
            props.outline_vertex_group = "ToonOutline"
        vg_name = props.outline_vertex_group
        count = 0
        target_obj = None
        for obj in context.selected_objects:
            if obj.type != 'MESH':
                continue
            vg = obj.vertex_groups.get(vg_name)
            if not vg:
                vg = obj.vertex_groups.new(name=vg_name)
            for v in obj.data.vertices:
                vg.add([v.index], 1.0, 'REPLACE')
            for mod in obj.modifiers:
                if mod.type == 'SOLIDIFY' and mod.name.startswith("ToonOutline"):
                    mod.vertex_group = vg_name
            count += 1
            target_obj = obj
        if count > 0:
            self.report({'INFO'}, bpy.app.translations.pgettext_iface("Created vertex group '{}' in {} objects").format(vg_name, count))
            try:
                if target_obj:
                    context.view_layer.objects.active = target_obj
                    bpy.ops.object.mode_set(mode='WEIGHT_PAINT')
            except Exception as e:
                print(f"OMOSEN+ Mode Switch Error: {e}")
        else:
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("Please select a mesh object"))
        return {'FINISHED'}
class OMOSEN_OT_SetupRenderForToon(Operator):
    """レンダー設定最適化"""
    bl_idname = "omosen.setup_render_for_toon"
    bl_label = "Set Render to Match Viewport"
    bl_description = "Apply settings to make render result match viewport"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        tr = bpy.app.translations.pgettext_iface
        engine_name, settings_applied = apply_render_optimizations(context)
        if len(settings_applied) > 0:
            self.report({'INFO'}, tr("Applied settings for {}: {}").format(engine_name, ', '.join(settings_applied)))
        else:
            self.report({'INFO'}, tr("All settings are already optimized"))
        return {'FINISHED'}
class OMOSEN_OT_SyncFromActive(Operator):
    """アクティブオブジェクトから設定を読み込む"""
    bl_idname = "omosen.sync_from_active"
    bl_label = "Load Settings from Active"
    bl_description = "Read toon settings from the active object into the UI"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        from ..core.sync import sync_props_from_active_object
        sync_props_from_active_object(context.scene)
        self.report({'INFO'}, bpy.app.translations.pgettext_iface("Settings loaded from object"))
        return {'FINISHED'}