"""コア - マテリアル操作（v1.1.3の値を維持）"""
import bpy
from .render_engine import get_engine_name
def _get_common_funcs():
    from . import materials_common
    return materials_common
def _is_cycles(context=None):
    return get_engine_name(context) == 'CYCLES'
def _get_cycles_module():
    from . import materials_cycles
    return materials_cycles
def _has_shader_to_rgb(mat: bpy.types.Material) -> bool:
    if not mat or not mat.use_nodes:
        return False
    return any(n.type == 'SHADER_TO_RGB' for n in mat.node_tree.nodes)
def _extract_base_color_and_image(mat: bpy.types.Material):
    base_color = (0.8, 0.8, 0.8, 1.0)
    image = None
    if not mat or not mat.use_nodes:
        return base_color, image
    for node in mat.node_tree.nodes:
        if node.label == "Toon_Original_Texture" and node.type == 'TEX_IMAGE' and node.image:
            image = node.image
        elif node.label == "Toon_Lighting_Diffuse" and node.type == 'BSDF_DIFFUSE':
            base_color = node.inputs[0].default_value[:]
    if image is None:
        for node in mat.node_tree.nodes:
            if node.type == 'BSDF_PRINCIPLED':
                base_input = node.inputs.get("Base Color")
                if base_input:
                    if base_input.links:
                        linked_node = base_input.links[0].from_node
                        if linked_node.type == 'TEX_IMAGE' and linked_node.image:
                            image = linked_node.image
                    else:
                        base_color = base_input.default_value[:]
                break
    return base_color, image
def _rebuild_cycles_material(mat: bpy.types.Material, props, cycles_module):
    if mat.name.startswith("ToonMaterial"):
        base_color, image = _extract_base_color_and_image(mat)
        cycles_module.rebuild_toon_material(mat, props, base_color, image)
    else:
        remove_toon_nodes_from_material(mat)
        cycles_module.insert_toon_nodes_to_material(mat, props)
def update_materials(context):
    """プロパティ変更時にマテリアルを更新（選択オブジェクトのみ）"""
    props = context.scene.omosen_props
    if props.is_syncing:
        return
    selected_objs = [obj for obj in context.selected_objects]
    if not selected_objs and context.active_object:
        selected_objs = [context.active_object]
    is_cycles = _is_cycles(context)
    cycles_module = _get_cycles_module() if is_cycles else None
    for obj in selected_objs:
        if obj.type == 'MESH':
            updated = False
            for slot in obj.material_slots:
                if not slot.material: continue
                if slot.material.use_nodes:
                    has_toon = any(n.label.startswith("Toon_") for n in slot.material.node_tree.nodes if n.label)
                    if has_toon:
                        if slot.material.users > 1:
                            slot.material = slot.material.copy()
                        if is_cycles:
                            cycles_module.update_existing_toon_nodes(slot.material, props)
                        else:
                            _update_existing_toon_nodes_eevee(slot.material, props)
                        updated = True
                if slot.material.name.startswith("OutlineMaterial"):
                    if slot.material.users > 1:
                        slot.material = slot.material.copy()
                    update_outline_material(slot.material, props)
                    updated = True
            for mod in obj.modifiers:
                if mod.type == 'SOLIDIFY' and mod.name.startswith("ToonOutline"):
                    if mod.thickness != props.outline_width:
                        mod.thickness = props.outline_width
                        updated = True
            if updated:
                obj.update_tag()
        elif obj.type == 'GPENCIL':
            if obj.name.startswith("ToonDrawing") or obj.name.startswith("ToonEdgeLines"):
                for mat in obj.data.materials:
                    if hasattr(mat, 'grease_pencil_settings'):
                        if mat.users > 1:
                            for i, m in enumerate(obj.data.materials):
                                if m == mat:
                                    new_gp_mat = mat.copy()
                                    obj.data.materials[i] = new_gp_mat
                                    mat = new_gp_mat
                                    break
                        new_color = (*props.edge_line_color, 1.0)
                        if mat.grease_pencil_settings.color != new_color:
                            mat.grease_pencil_settings.color = new_color
                obj.update_tag()
    if context.area:
        context.area.tag_redraw()
def update_ramp_nodes(ramp_node, props):
    """ColorRampノードの影設定を更新"""
    common = _get_common_funcs()
    return common.update_ramp_nodes(ramp_node, props)
def setup_two_level_shadows(ramp, props):
    """2段階の影を設定"""
    common = _get_common_funcs()
    return common.setup_two_level_shadows(ramp, props)
def setup_three_level_shadows(ramp, props):
    """3段階の影を設定"""
    common = _get_common_funcs()
    return common.setup_three_level_shadows(ramp, props)
def update_outline_material(mat: bpy.types.Material, props):
    """アウトラインマテリアルのノードを更新"""
    common = _get_common_funcs()
    return common.update_outline_material(mat, props)
def create_outline_material(props) -> bpy.types.Material:
    """アウトライン用マテリアルを作成（v1.1.3）"""
    common = _get_common_funcs()
    return common.create_outline_material(props)
def create_toon_material(props) -> bpy.types.Material:
    """トゥーンシェーディング用マテリアルを作成（Eevee/Cycles分岐）"""
    if _is_cycles():
        return _get_cycles_module().create_toon_material(props)
    return _create_toon_material_eevee(props)
def _create_toon_material_eevee(props) -> bpy.types.Material:
    """Eevee用トゥーンシェーディング用マテリアルを作成"""
    name = "ToonMaterial"
    if name in bpy.data.materials:
        mat = bpy.data.materials[name]
    else:
        mat = bpy.data.materials.new(name)
    mat.use_nodes = True
    mat.blend_method = 'OPAQUE'
    if hasattr(mat, 'shadow_method'):
        mat.shadow_method = 'OPAQUE'
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    nodes.clear()
    diffuse = nodes.new("ShaderNodeBsdfDiffuse")
    diffuse.inputs[0].default_value = (1.0, 1.0, 1.0, 1.0)
    diffuse.inputs[1].default_value = 0.0
    diffuse.location = (-1200, 200)
    diffuse.label = "Toon_Lighting_Diffuse"
    shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    shader_to_rgb.location = (-900, 200)
    shader_to_rgb.label = "Toon_ShaderToRGB"
    geom = nodes.new("ShaderNodeNewGeometry")
    geom.location = (-1200, -200)
    geom.label = "Toon_Geometry"
    light_vec = nodes.new("ShaderNodeRGB")
    light_vec.location = (-1200, -650)
    light_vec.label = "Toon_LightVector"
    light_vec.outputs[0].default_value = (*props.light_direction, 1.0)
    dot = nodes.new("ShaderNodeVectorMath")
    dot.operation = 'DOT_PRODUCT'
    dot.location = (-900, -200)
    dot.label = "Toon_DotProduct"
    dot_add = nodes.new("ShaderNodeMath")
    dot_add.operation = 'ADD'
    dot_add.inputs[1].default_value = 1.0
    dot_add.location = (-650, -200)
    dot_add.label = "Toon_DotAdd"
    dot_mul = nodes.new("ShaderNodeMath")
    dot_mul.operation = 'MULTIPLY'
    dot_mul.inputs[1].default_value = 0.5
    dot_mul.location = (-400, -200)
    dot_mul.label = "Toon_DotMul"
    shading_mix = nodes.new("ShaderNodeMixRGB")
    shading_mix.blend_type = 'MIX'
    shading_mix.inputs[0].default_value = 1.0 if props.shading_method == 'VECTOR' else 0.0
    shading_mix.location = (-400, 100)
    shading_mix.label = "Toon_ShadingMix"
    math_multiply = nodes.new("ShaderNodeMath")
    math_multiply.operation = 'MULTIPLY'
    math_multiply.inputs[1].default_value = props.light_boost
    math_multiply.location = (-150, 0)
    math_multiply.label = "Toon_LightBoost"
    math_hardness = nodes.new("ShaderNodeMapRange")
    math_hardness.data_type = 'FLOAT'
    math_hardness.interpolation_type = 'LINEAR'
    math_hardness.location = (100, 0)
    math_hardness.label = "Toon_Hardness"
    h_val = 1.0 / max(props.shadow_hardness, 0.001)
    math_hardness.inputs[1].default_value = props.shadow_step - h_val
    math_hardness.inputs[2].default_value = props.shadow_step + h_val
    math_hardness.inputs[3].default_value = 0.0
    math_hardness.inputs[4].default_value = 1.0
    ramp = nodes.new("ShaderNodeValToRGB")
    ramp.location = (400, 0)
    ramp.label = "Toon_ColorRamp"
    update_ramp_nodes(ramp, props)
    glossy = nodes.new("ShaderNodeBsdfGlossy")
    glossy.inputs[1].default_value = 0.0
    glossy.location = (100, -350)
    glossy.label = "Toon_Glossy"
    spec_shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    spec_shader_to_rgb.location = (350, -350)
    spec_shader_to_rgb.label = "Toon_SpecShaderToRGB"
    spec_ramp = nodes.new("ShaderNodeValToRGB")
    spec_ramp.color_ramp.interpolation = 'CONSTANT'
    spec_ramp.location = (600, -350)
    spec_ramp.label = "Toon_SpecRamp"
    spec_ramp.color_ramp.elements[0].position = 1.0 - props.specular_size
    spec_ramp.color_ramp.elements[0].color = (0, 0, 0, 1)
    if len(spec_ramp.color_ramp.elements) < 2:
        spec_ramp.color_ramp.elements.new(1.0)
    spec_ramp.color_ramp.elements[1].position = 1.0 - props.specular_size + 0.001
    spec_ramp.color_ramp.elements[1].color = (1, 1, 1, 1)
    spec_mix = nodes.new("ShaderNodeMixRGB")
    spec_mix.blend_type = 'MIX'
    spec_mix.inputs[1].default_value = (0, 0, 0, 1)
    spec_mix.inputs[2].default_value = (*props.specular_color, 1.0)
    spec_mix.location = (1100, -350)
    spec_mix.label = "Toon_SpecMix"
    fresnel = nodes.new("ShaderNodeFresnel")
    fresnel.location = (100, -700)
    fresnel.label = "Toon_Fresnel"
    rim_map = nodes.new("ShaderNodeMapRange")
    rim_map.data_type = 'FLOAT'
    rim_map.interpolation_type = 'LINEAR'
    rim_map.location = (350, -700)
    rim_map.label = "Toon_RimMap"
    fresnel.inputs[0].default_value = 1.45
    rim_map.inputs[1].default_value = 1.0 - props.rim_light_width
    rim_map.inputs[2].default_value = 1.0 - props.rim_light_width + 0.05
    rim_map.inputs[3].default_value = 0.0
    rim_map.inputs[4].default_value = 1.0
    rim_mix = nodes.new("ShaderNodeMixRGB")
    rim_mix.blend_type = 'MIX'
    rim_mix.inputs[1].default_value = (0, 0, 0, 1)
    rim_mix.inputs[2].default_value = (*props.rim_light_color, 1.0)
    rim_mix.location = (850, -700)
    rim_mix.label = "Toon_RimMix"
    mix_color = nodes.new("ShaderNodeMixRGB")
    mix_color.blend_type = 'MULTIPLY'
    mix_color.inputs[0].default_value = props.shadow_strength
    mix_color.inputs[1].default_value = (*props.highlight_color, 1.0)
    mix_color.location = (750, 200)
    mix_color.label = "Toon_Mix"
    add_rim = nodes.new("ShaderNodeMixRGB")
    add_rim.blend_type = 'ADD'
    add_rim.inputs[0].default_value = props.rim_light_strength if props.use_rim_light else 0.0
    add_rim.location = (1000, 200)
    add_rim.label = "Toon_AddRim"
    add_spec = nodes.new("ShaderNodeMixRGB")
    add_spec.blend_type = 'ADD'
    add_spec.inputs[0].default_value = 1.0 if props.use_specular else 0.0
    add_spec.location = (1250, 200)
    add_spec.label = "Toon_AddSpec"
    emission = nodes.new("ShaderNodeEmission")
    emission.inputs[1].default_value = 1.0
    emission.location = (1500, 200)
    emission.label = "Toon_Emission"
    output = nodes.new("ShaderNodeOutputMaterial")
    output.location = (1800, 200)
    links.new(diffuse.outputs[0], shader_to_rgb.inputs[0])
    links.new(geom.outputs["Normal"], dot.inputs[0])
    links.new(light_vec.outputs[0], dot.inputs[1])
    links.new(dot.outputs["Value"], dot_add.inputs[0])
    links.new(dot_add.outputs[0], dot_mul.inputs[0])
    links.new(shader_to_rgb.outputs[0], shading_mix.inputs[1])
    links.new(dot_mul.outputs[0], shading_mix.inputs[2])
    links.new(shading_mix.outputs[0], math_multiply.inputs[0])
    links.new(math_multiply.outputs[0], math_hardness.inputs[0])
    links.new(math_hardness.outputs[0], ramp.inputs[0])
    links.new(glossy.outputs[0], spec_shader_to_rgb.inputs[0])
    links.new(spec_shader_to_rgb.outputs[0], spec_ramp.inputs[0])
    links.new(spec_ramp.outputs[0], spec_mix.inputs[0])
    links.new(fresnel.outputs[0], rim_map.inputs[0])
    links.new(rim_map.outputs[0], rim_mix.inputs[0])
    links.new(mix_color.outputs[0], add_rim.inputs[1])
    links.new(ramp.outputs[0], mix_color.inputs[2])
    links.new(rim_mix.outputs[0], add_rim.inputs[2])
    links.new(add_rim.outputs[0], add_spec.inputs[1])
    links.new(spec_mix.outputs[0], add_spec.inputs[2])
    links.new(add_spec.outputs[0], emission.inputs[0])
    links.new(emission.outputs[0], output.inputs[0])
    return mat
def create_toon_material_from_original(original_mat: bpy.types.Material, props, obj_name: str) -> bpy.types.Material:
    """元のマテリアルのテクスチャ/ベースカラーを保持しながらトゥーン化（Eevee/Cycles分岐）"""
    if _is_cycles():
        return _get_cycles_module().create_toon_material_from_original(original_mat, props, obj_name)
    return _create_toon_material_from_original_eevee(original_mat, props, obj_name)
def _create_toon_material_from_original_eevee(original_mat: bpy.types.Material, props, obj_name: str) -> bpy.types.Material:
    """Eevee用: 元のマテリアルのテクスチャ/ベースカラーを保持しながらトゥーン化"""
    name = f"ToonMaterial_{obj_name}"
    if name in bpy.data.materials:
        mat = bpy.data.materials[name]
    else:
        mat = bpy.data.materials.new(name)
    mat.use_nodes = True
    mat.blend_method = 'OPAQUE'
    if hasattr(mat, 'shadow_method'):
        mat.shadow_method = 'OPAQUE'
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    nodes.clear()
    base_color = (0.8, 0.8, 0.8, 1.0)
    texture_node = None
    if original_mat and original_mat.use_nodes:
        for node in original_mat.node_tree.nodes:
            if node.type == 'BSDF_PRINCIPLED':
                base_color_input = node.inputs.get("Base Color")
                if base_color_input:
                    if base_color_input.links:
                        linked_node = base_color_input.links[0].from_node
                        if linked_node.type == 'TEX_IMAGE' and linked_node.image:
                            texture_node = nodes.new("ShaderNodeTexImage")
                            texture_node.image = linked_node.image
                            texture_node.location = (-1000, 200)
                            texture_node.label = "Toon_Original_Texture"
                    else:
                        base_color = base_color_input.default_value[:]
                break
    diffuse = nodes.new("ShaderNodeBsdfDiffuse")
    diffuse.inputs[1].default_value = 0.0
    diffuse.location = (-800, 0)
    diffuse.label = "Toon_Lighting_Diffuse"
    if texture_node:
        links.new(texture_node.outputs[0], diffuse.inputs[0])
    else:
        diffuse.inputs[0].default_value = base_color
    shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    shader_to_rgb.location = (-600, 0)
    shader_to_rgb.label = "Toon_ShaderToRGB"
    geom = nodes.new("ShaderNodeNewGeometry")
    geom.location = (-1000, -200)
    geom.label = "Toon_Geometry"
    light_vec = nodes.new("ShaderNodeRGB")
    light_vec.location = (-1000, -400)
    light_vec.label = "Toon_LightVector"
    light_vec.outputs[0].default_value = (*props.light_direction, 1.0)
    dot = nodes.new("ShaderNodeVectorMath")
    dot.operation = 'DOT_PRODUCT'
    dot.location = (-800, -200)
    dot.label = "Toon_DotProduct"
    dot_add = nodes.new("ShaderNodeMath")
    dot_add.operation = 'ADD'
    dot_add.inputs[1].default_value = 1.0
    dot_add.location = (-600, -200)
    dot_add.label = "Toon_DotAdd"
    dot_mul = nodes.new("ShaderNodeMath")
    dot_mul.operation = 'MULTIPLY'
    dot_mul.inputs[1].default_value = 0.5
    dot_mul.location = (-400, -200)
    dot_mul.label = "Toon_DotMul"
    shading_mix = nodes.new("ShaderNodeMixRGB")
    shading_mix.blend_type = 'MIX'
    shading_mix.inputs[0].default_value = 1.0 if props.shading_method == 'VECTOR' else 0.0
    shading_mix.location = (-200, 0)
    shading_mix.label = "Toon_ShadingMix"
    math_multiply = nodes.new("ShaderNodeMath")
    math_multiply.operation = 'MULTIPLY'
    math_multiply.inputs[1].default_value = props.light_boost
    math_multiply.location = (-400, 0)
    math_multiply.label = "Toon_LightBoost"
    math_hardness = nodes.new("ShaderNodeMapRange")
    math_hardness.data_type = 'FLOAT'
    math_hardness.interpolation_type = 'LINEAR'
    math_hardness.location = (200, 0)
    math_hardness.label = "Toon_Hardness"
    h_val = 1.0 / max(props.shadow_hardness, 0.001)
    math_hardness.inputs[1].default_value = props.shadow_step - h_val
    math_hardness.inputs[2].default_value = props.shadow_step + h_val
    math_hardness.inputs[3].default_value = 0.0
    math_hardness.inputs[4].default_value = 1.0
    ramp = nodes.new("ShaderNodeValToRGB")
    ramp.location = (400, -100)
    ramp.label = "Toon_ColorRamp"
    update_ramp_nodes(ramp, props)
    glossy = nodes.new("ShaderNodeBsdfGlossy")
    glossy.inputs[1].default_value = 0.0
    glossy.location = (0, -300)
    glossy.label = "Toon_Glossy"
    spec_shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    spec_shader_to_rgb.location = (200, -300)
    spec_shader_to_rgb.label = "Toon_SpecShaderToRGB"
    spec_ramp = nodes.new("ShaderNodeValToRGB")
    spec_ramp.color_ramp.interpolation = 'CONSTANT'
    spec_ramp.location = (400, -300)
    spec_ramp.label = "Toon_SpecRamp"
    spec_ramp.color_ramp.elements[0].position = 1.0 - props.specular_size
    spec_ramp.color_ramp.elements[0].color = (0, 0, 0, 1)
    if len(spec_ramp.color_ramp.elements) < 2:
        spec_ramp.color_ramp.elements.new(1.0)
    spec_ramp.color_ramp.elements[1].position = 1.0 - props.specular_size + 0.001
    spec_ramp.color_ramp.elements[1].color = (1, 1, 1, 1)
    spec_mix = nodes.new("ShaderNodeMixRGB")
    spec_mix.blend_type = 'MIX'
    spec_mix.inputs[1].default_value = (0, 0, 0, 1)
    spec_mix.inputs[2].default_value = (*props.specular_color, 1.0)
    spec_mix.location = (600, -300)
    spec_mix.label = "Toon_SpecMix"
    fresnel = nodes.new("ShaderNodeFresnel")
    fresnel.location = (0, -600)
    fresnel.label = "Toon_Fresnel"
    rim_map = nodes.new("ShaderNodeMapRange")
    rim_map.data_type = 'FLOAT'
    rim_map.interpolation_type = 'LINEAR'
    rim_map.location = (200, -600)
    rim_map.label = "Toon_RimMap"
    fresnel.inputs[0].default_value = 1.45
    rim_map.inputs[1].default_value = 1.0 - props.rim_light_width
    rim_map.inputs[2].default_value = 1.0 - props.rim_light_width + 0.05
    rim_map.inputs[3].default_value = 0.0
    rim_map.inputs[4].default_value = 1.0
    rim_mix = nodes.new("ShaderNodeMixRGB")
    rim_mix.blend_type = 'MIX'
    rim_mix.inputs[1].default_value = (0, 0, 0, 1)
    rim_mix.inputs[2].default_value = (*props.rim_light_color, 1.0)
    rim_mix.location = (400, -600)
    rim_mix.label = "Toon_RimMix"
    mix_color = nodes.new("ShaderNodeMixRGB")
    mix_color.blend_type = 'MULTIPLY'
    mix_color.inputs[0].default_value = props.shadow_strength
    mix_color.location = (600, 0)
    mix_color.label = "Toon_Mix"
    add_rim = nodes.new("ShaderNodeMixRGB")
    add_rim.blend_type = 'ADD'
    add_rim.inputs[0].default_value = props.rim_light_strength if props.use_rim_light else 0.0
    add_rim.location = (800, -100)
    add_rim.label = "Toon_AddRim"
    add_spec = nodes.new("ShaderNodeMixRGB")
    add_spec.blend_type = 'ADD'
    add_spec.inputs[0].default_value = 1.0 if props.use_specular else 0.0
    add_spec.location = (1000, 0)
    add_spec.label = "Toon_AddSpec"
    emission = nodes.new("ShaderNodeEmission")
    emission.inputs[1].default_value = 1.0
    emission.location = (1000, 0)
    emission.label = "Toon_Emission"
    output = nodes.new("ShaderNodeOutputMaterial")
    output.location = (1200, 0)
    links.new(diffuse.outputs[0], shader_to_rgb.inputs[0])
    links.new(geom.outputs["Normal"], dot.inputs[0])
    links.new(light_vec.outputs[0], dot.inputs[1])
    links.new(dot.outputs["Value"], dot_add.inputs[0])
    links.new(dot_add.outputs[0], dot_mul.inputs[0])
    links.new(shader_to_rgb.outputs[0], shading_mix.inputs[1])
    links.new(dot_mul.outputs[0], shading_mix.inputs[2])
    links.new(shading_mix.outputs[0], math_multiply.inputs[0])
    links.new(math_multiply.outputs[0], math_hardness.inputs[0])
    links.new(math_hardness.outputs[0], ramp.inputs[0])
    links.new(glossy.outputs[0], spec_shader_to_rgb.inputs[0])
    links.new(spec_shader_to_rgb.outputs[0], spec_ramp.inputs[0])
    links.new(spec_ramp.outputs[0], spec_mix.inputs[0])
    links.new(fresnel.outputs[0], rim_map.inputs[0])
    links.new(rim_map.outputs[0], rim_mix.inputs[0])
    if texture_node:
        links.new(texture_node.outputs[0], mix_color.inputs[1])
    else:
        mix_color.inputs[1].default_value = base_color
    links.new(ramp.outputs[0], mix_color.inputs[2])
    links.new(mix_color.outputs[0], add_rim.inputs[1])
    links.new(rim_mix.outputs[0], add_rim.inputs[2])
    links.new(add_rim.outputs[0], add_spec.inputs[1])
    links.new(spec_mix.outputs[0], add_spec.inputs[2])
    links.new(add_spec.outputs[0], emission.inputs[0])
    links.new(emission.outputs[0], output.inputs[0])
    return mat
def insert_toon_nodes_to_material(mat: bpy.types.Material, props) -> bool:
    """既存マテリアルにトゥーンノードを挿入（非破壊 / Eevee/Cycles分岐）"""
    if _is_cycles():
        if _has_shader_to_rgb(mat):
            if mat.name.startswith("ToonMaterial"):
                base_color, image = _extract_base_color_and_image(mat)
                _get_cycles_module().rebuild_toon_material(mat, props, base_color, image)
                return True
            remove_toon_nodes_from_material(mat)
        return _get_cycles_module().insert_toon_nodes_to_material(mat, props)
    return _insert_toon_nodes_to_material_eevee(mat, props)
def _insert_toon_nodes_to_material_eevee(mat: bpy.types.Material, props) -> bool:
    """Eevee用: 既存マテリアルにトゥーンノードを挿入（非破壊）"""
    if not mat or not mat.use_nodes:
        return False
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    for node in nodes:
        if node.label == "Toon_Base_ShaderToRGB":
            return update_existing_toon_nodes(mat, props)
    principled = None
    output = None
    for node in nodes:
        if node.type == 'BSDF_PRINCIPLED':
            principled = node
        elif node.type == 'OUTPUT_MATERIAL':
            output = node
    if not principled or not output:
        return False
    original_link = None
    for link in links:
        if link.from_node == principled and link.to_node == output:
            original_link = link
            break
    if not original_link:
        return False
    links.remove(original_link)
    x_start = principled.location.x + 300
    base_shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    base_shader_to_rgb.location = (x_start, principled.location.y + 300)
    base_shader_to_rgb.label = "Toon_Base_ShaderToRGB"
    diffuse = nodes.new("ShaderNodeBsdfDiffuse")
    diffuse.inputs[0].default_value = (1.0, 1.0, 1.0, 1.0)
    diffuse.inputs[1].default_value = 0.0
    diffuse.location = (x_start, principled.location.y)
    diffuse.label = "Toon_Lighting_Diffuse"
    lighting_shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    lighting_shader_to_rgb.location = (x_start + 200, principled.location.y)
    lighting_shader_to_rgb.label = "Toon_Lighting_ShaderToRGB"
    geom = nodes.new("ShaderNodeNewGeometry")
    geom.location = (x_start, principled.location.y - 300)
    geom.label = "Toon_Geometry"
    light_vec = nodes.new("ShaderNodeRGB")
    light_vec.location = (x_start + 5, principled.location.y - 548)
    light_vec.label = "Toon_LightVector"
    light_vec.outputs[0].default_value = (*props.light_direction, 1.0)
    dot = nodes.new("ShaderNodeVectorMath")
    dot.operation = 'DOT_PRODUCT'
    dot.location = (x_start + 200, principled.location.y - 300)
    dot.label = "Toon_DotProduct"
    dot_add = nodes.new("ShaderNodeMath")
    dot_add.operation = 'ADD'
    dot_add.inputs[1].default_value = 1.0
    dot_add.location = (x_start + 400, principled.location.y - 300)
    dot_add.label = "Toon_DotAdd"
    dot_mul = nodes.new("ShaderNodeMath")
    dot_mul.operation = 'MULTIPLY'
    dot_mul.inputs[1].default_value = 0.5
    dot_mul.location = (x_start + 600, principled.location.y - 300)
    dot_mul.label = "Toon_DotMul"
    shading_mix = nodes.new("ShaderNodeMixRGB")
    shading_mix.blend_type = 'MIX'
    shading_mix.inputs[0].default_value = 1.0 if props.shading_method == 'VECTOR' else 0.0
    shading_mix.location = (x_start + 810, principled.location.y)
    shading_mix.label = "Toon_ShadingMix"
    math_multiply = nodes.new("ShaderNodeMath")
    math_multiply.operation = 'MULTIPLY'
    math_multiply.inputs[1].default_value = props.light_boost
    math_multiply.location = (x_start + 1050, principled.location.y)
    math_multiply.label = "Toon_LightBoost"
    math_hardness = nodes.new("ShaderNodeMapRange")
    math_hardness.data_type = 'FLOAT'
    math_hardness.interpolation_type = 'LINEAR'
    math_hardness.location = (x_start + 1300, principled.location.y)
    math_hardness.label = "Toon_Hardness"
    h_val = 1.0 / max(props.shadow_hardness, 0.001)
    math_hardness.inputs[1].default_value = props.shadow_step - h_val
    math_hardness.inputs[2].default_value = props.shadow_step + h_val
    math_hardness.inputs[3].default_value = 0.0
    math_hardness.inputs[4].default_value = 1.0
    ramp = nodes.new("ShaderNodeValToRGB")
    ramp.location = (x_start + 1500, principled.location.y)
    ramp.label = "Toon_ColorRamp"
    update_ramp_nodes(ramp, props)
    glossy = nodes.new("ShaderNodeBsdfGlossy")
    glossy.inputs[1].default_value = 0.0
    glossy.location = (x_start + 1000, principled.location.y - 300)
    glossy.label = "Toon_Glossy"
    spec_shader_to_rgb = nodes.new("ShaderNodeShaderToRGB")
    spec_shader_to_rgb.location = (x_start + 1200, principled.location.y - 300)
    spec_shader_to_rgb.label = "Toon_SpecShaderToRGB"
    spec_ramp = nodes.new("ShaderNodeValToRGB")
    spec_ramp.color_ramp.interpolation = 'CONSTANT'
    spec_ramp.location = (x_start + 1400, principled.location.y - 300)
    spec_ramp.label = "Toon_SpecRamp"
    spec_ramp.color_ramp.elements[0].position = 1.0 - props.specular_size
    spec_ramp.color_ramp.elements[0].color = (0, 0, 0, 1)
    if len(spec_ramp.color_ramp.elements) < 2:
        spec_ramp.color_ramp.elements.new(1.0)
    spec_ramp.color_ramp.elements[1].position = 1.0 - props.specular_size + 0.001
    spec_ramp.color_ramp.elements[1].color = (1, 1, 1, 1)
    spec_mix = nodes.new("ShaderNodeMixRGB")
    spec_mix.blend_type = 'MIX'
    spec_mix.inputs[1].default_value = (0, 0, 0, 1)
    spec_mix.inputs[2].default_value = (*props.specular_color, 1.0)
    spec_mix.location = (x_start + 1910, principled.location.y - 300)
    spec_mix.label = "Toon_SpecMix"
    fresnel = nodes.new("ShaderNodeFresnel")
    fresnel.location = (x_start + 1200, principled.location.y - 600)
    fresnel.label = "Toon_Fresnel"
    rim_map = nodes.new("ShaderNodeMapRange")
    rim_map.data_type = 'FLOAT'
    rim_map.interpolation_type = 'LINEAR'
    rim_map.location = (x_start + 1400, principled.location.y - 600)
    rim_map.label = "Toon_RimMap"
    fresnel.inputs[0].default_value = 1.45
    rim_map.inputs[1].default_value = 1.0 - props.rim_light_width
    rim_map.inputs[2].default_value = 1.0 - props.rim_light_width + 0.05
    rim_map.inputs[3].default_value = 0.0
    rim_map.inputs[4].default_value = 1.0
    rim_mix = nodes.new("ShaderNodeMixRGB")
    rim_mix.blend_type = 'MIX'
    rim_mix.inputs[1].default_value = (0, 0, 0, 1)
    rim_mix.inputs[2].default_value = (*props.rim_light_color, 1.0)
    rim_mix.location = (x_start + 1600, principled.location.y - 600)
    rim_mix.label = "Toon_RimMix"
    mix_color = nodes.new("ShaderNodeMixRGB")
    mix_color.blend_type = 'MULTIPLY'
    mix_color.inputs[0].default_value = props.shadow_strength
    mix_color.location = (x_start + 1800, principled.location.y + 300)
    mix_color.label = "Toon_Mix"
    add_rim = nodes.new("ShaderNodeMixRGB")
    add_rim.blend_type = 'ADD'
    add_rim.inputs[0].default_value = props.rim_light_strength if props.use_rim_light else 0.0
    add_rim.location = (x_start + 2045, principled.location.y + 301)
    add_rim.label = "Toon_AddRim"
    add_spec = nodes.new("ShaderNodeMixRGB")
    add_spec.blend_type = 'ADD'
    add_spec.inputs[0].default_value = 1.0 if props.use_specular else 0.0
    add_spec.location = (x_start + 2250, principled.location.y + 300)
    add_spec.label = "Toon_AddSpec"
    emission = nodes.new("ShaderNodeEmission")
    emission.inputs[1].default_value = 1.0
    emission.location = (x_start + 2450, principled.location.y + 300)
    emission.label = "Toon_Emission"
    output.location = (x_start + 2700, principled.location.y + 300)
    links.new(principled.outputs[0], base_shader_to_rgb.inputs[0])
    links.new(diffuse.outputs[0], lighting_shader_to_rgb.inputs[0])
    links.new(geom.outputs["Normal"], dot.inputs[0])
    links.new(light_vec.outputs[0], dot.inputs[1])
    links.new(dot.outputs["Value"], dot_add.inputs[0])
    links.new(dot_add.outputs[0], dot_mul.inputs[0])
    links.new(lighting_shader_to_rgb.outputs[0], shading_mix.inputs[1])
    links.new(dot_mul.outputs[0], shading_mix.inputs[2])
    links.new(shading_mix.outputs[0], math_multiply.inputs[0])
    links.new(math_multiply.outputs[0], math_hardness.inputs[0])
    links.new(math_hardness.outputs[0], ramp.inputs[0])
    links.new(glossy.outputs[0], spec_shader_to_rgb.inputs[0])
    links.new(spec_shader_to_rgb.outputs[0], spec_ramp.inputs[0])
    links.new(spec_ramp.outputs[0], spec_mix.inputs[0])
    links.new(fresnel.outputs[0], rim_map.inputs[0])
    links.new(rim_map.outputs[0], rim_mix.inputs[0])
    if principled.inputs['Base Color'].is_linked:
        links.new(principled.inputs['Base Color'].links[0].from_socket, mix_color.inputs[1])
    else:
        mix_color.inputs[1].default_value = principled.inputs['Base Color'].default_value
    links.new(ramp.outputs[0], mix_color.inputs[2])
    links.new(mix_color.outputs[0], add_rim.inputs[1])
    links.new(rim_mix.outputs[0], add_rim.inputs[2])
    links.new(add_rim.outputs[0], add_spec.inputs[1])
    links.new(spec_mix.outputs[0], add_spec.inputs[2])
    links.new(add_spec.outputs[0], emission.inputs[0])
    links.new(emission.outputs[0], output.inputs[0])
    return True
def update_existing_toon_nodes(mat: bpy.types.Material, props) -> bool:
    """既存のトゥーンノードを更新（Eevee/Cycles分岐）"""
    if _is_cycles():
        return _get_cycles_module().update_existing_toon_nodes(mat, props)
    return _update_existing_toon_nodes_eevee(mat, props)
def _update_existing_toon_nodes_eevee(mat: bpy.types.Material, props) -> bool:
    """Eevee用: 既存のトゥーンノードを更新"""
    if not mat or not mat.use_nodes:
        return False
    nodes = mat.node_tree.nodes
    for node in nodes:
        id_str = node.label if node.label else node.name
        if not id_str: continue
        ntype = node.type
        is_mix_type = ntype in {'MIX_RGB', 'MIX'}
        is_mix_node = (ntype == 'MIX')
        if "Toon_LightBoost" in id_str and ntype == 'MATH':
            node.inputs[1].default_value = props.light_boost
        elif "Toon_Hardness" in id_str:
            if ntype == 'MAP_RANGE':
                h_val = 1.0 / max(props.shadow_hardness, 0.001)
                node.inputs[1].default_value = props.shadow_step - h_val
                node.inputs[2].default_value = props.shadow_step + h_val
            elif ntype == 'MATH':
                node.inputs[1].default_value = props.shadow_hardness
        elif "Toon_Mix" in id_str and is_mix_type:
            node.inputs[0].default_value = props.shadow_strength
        elif "Toon_AddSpec" in id_str and is_mix_type:
            node.inputs[0].default_value = 1.0 if props.use_specular else 0.0
        elif "Toon_SpecMix" in id_str and is_mix_type:
            if is_mix_node:
                soc = node.inputs.get("B") or node.inputs.get("Color2") or (node.inputs[7] if len(node.inputs) > 7 else None)
                if soc: soc.default_value = (*props.specular_color, 1.0)
            else:
                node.inputs[2].default_value = (*props.specular_color, 1.0)
        elif "Toon_SpecRamp" in id_str and ntype == 'VALTORGB':
            node.color_ramp.elements[0].position = 1.0 - props.specular_size
            if len(node.color_ramp.elements) > 1:
                node.color_ramp.elements[1].position = 1.0 - props.specular_size + 0.001
        elif "Toon_ColorRamp" in id_str and ntype == 'VALTORGB':
            update_ramp_nodes(node, props)
        elif "Toon_RimMap" in id_str and ntype == 'MAP_RANGE':
            node.inputs[1].default_value = 1.0 - props.rim_light_width
            node.inputs[2].default_value = 1.0 - props.rim_light_width + 0.05
        elif "Toon_RimMix" in id_str and is_mix_type:
            if is_mix_node:
                soc = node.inputs.get("B") or node.inputs.get("Color2") or (node.inputs[7] if len(node.inputs) > 7 else None)
                if soc: soc.default_value = (*props.rim_light_color, 1.0)
            else:
                node.inputs[2].default_value = (*props.rim_light_color, 1.0)
        elif "Toon_AddRim" in id_str and is_mix_type:
            node.inputs[0].default_value = props.rim_light_strength if props.use_rim_light else 0.0
        elif "Toon_ShadingMix" in id_str and is_mix_type:
            node.inputs[0].default_value = 1.0 if props.shading_method == 'VECTOR' else 0.0
        elif "Toon_LightVector" in id_str:
            node.outputs[0].default_value = (*props.light_direction, 1.0)
    return True
def remove_toon_nodes_from_material(mat: bpy.types.Material) -> bool:
    """マテリアルからトゥーンノードを削除し、元の接続を復元
    INSERTモードで挿入されたノードを削除し、
    Principled BSDF → Material Output の接続を復元する
    """
    if not mat or not mat.use_nodes:
        return False
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    toon_nodes = []
    principled = None
    output = None
    for node in nodes:
        if (node.label and node.label.startswith("Toon_")) or (node.name and node.name.startswith("Toon_")):
            toon_nodes.append(node)
        elif node.type == 'BSDF_PRINCIPLED':
            principled = node
        elif node.type == 'OUTPUT_MATERIAL':
            output = node
    if not toon_nodes:
        return False
    for node in toon_nodes:
        nodes.remove(node)
    if principled and output:
        has_connection = any(link.to_node == output and link.to_socket.name == "Surface" for link in links)
        if not has_connection:
            links.new(principled.outputs["BSDF"], output.inputs["Surface"])
    return True