"""コア - マテリアル共通処理（Eevee/Cycles共通）"""
import bpy
def _is_cycles():
    try:
        return bpy.context.scene.render.engine == 'CYCLES'
    except Exception:
        return False
def update_ramp_nodes(ramp_node, props):
    """ColorRampノードの影設定を更新"""
    if props.shadow_levels == 2:
        setup_two_level_shadows(ramp_node, props)
    elif props.shadow_levels == 3:
        setup_three_level_shadows(ramp_node, props)
def setup_two_level_shadows(ramp, props):
    """2段階の影を設定"""
    r = ramp.color_ramp
    r.interpolation = props.interpolation
    while len(r.elements) > 2:
        r.elements.remove(r.elements[-1])
    while len(r.elements) < 2:
        r.elements.new(0.5)
    r.elements[0].position = props.shadow_step
    r.elements[0].color = (*props.shadow_color, 1.0)
    if props.interpolation == 'CONSTANT':
        r.elements[1].position = min(props.shadow_step + 0.001, 1.0)
    else:
        r.elements[1].position = min(props.shadow_step + 0.2, 1.0)
    r.elements[1].color = (*props.highlight_color, 1.0)
def setup_three_level_shadows(ramp, props):
    """3段階の影を設定"""
    r = ramp.color_ramp
    r.interpolation = props.interpolation
    while len(r.elements) > 3:
        r.elements.remove(r.elements[-1])
    while len(r.elements) < 3:
        r.elements.new(0.5)
    r.elements[0].position = 0.0
    r.elements[0].color = (*props.shadow_color_2, 1.0)
    if props.interpolation == 'CONSTANT':
        pos1 = max(props.shadow_step - 0.2, 0.01)
    else:
        pos1 = props.shadow_step
    r.elements[1].position = pos1
    r.elements[1].color = (*props.shadow_color, 1.0)
    if props.interpolation == 'CONSTANT':
        pos2 = max(props.shadow_step, pos1 + 0.01)
    else:
        pos2 = min(props.shadow_step + 0.2, 1.0)
    r.elements[2].position = pos2
    r.elements[2].color = (*props.highlight_color, 1.0)
def update_outline_material(mat: bpy.types.Material, props):
    """アウトラインマテリアルのノードを更新"""
    if not mat.use_nodes:
        return
    nodes = mat.node_tree.nodes
    has_shadow_blocking = any(n.type == 'LIGHT_PATH' for n in nodes)
    has_backfacing = any(n.label == "Toon_Backfacing" for n in nodes)
    if not has_shadow_blocking or (_is_cycles() and not has_backfacing):
        create_outline_material(props)
        return
    target_node = None
    for node in nodes:
        if node.label == "Toon_OutlineEmission" and node.type == 'EMISSION':
            target_node = node
            break
    if not target_node:
        for node in nodes:
            if node.type == 'EMISSION':
                target_node = node
                break
    if target_node:
        target_node.inputs["Color"].default_value = (*props.outline_color, 1.0)
    if hasattr(mat, "shadow_method"):
        mat.shadow_method = 'NONE'
    if hasattr(mat, "use_transparent_shadow"):
        mat.use_transparent_shadow = True
    if hasattr(mat, "surface_render_method"):
        if mat.surface_render_method == 'OPAQUE':
            mat.surface_render_method = 'HASHED'
    elif hasattr(mat, "blend_method"):
        if mat.blend_method == 'OPAQUE':
            mat.blend_method = 'HASHED'
def create_outline_material(props) -> bpy.types.Material:
    """アウトライン用マテリアルを作成（Eevee/Cycles共通）"""
    name = "OutlineMaterial"
    if name in bpy.data.materials:
        mat = bpy.data.materials[name]
    else:
        mat = bpy.data.materials.new(name)
    mat.use_nodes = True
    mat.use_backface_culling = True
    if hasattr(mat, "shadow_method"):
        mat.shadow_method = 'NONE'
    if hasattr(mat, "use_transparent_shadow"):
        mat.use_transparent_shadow = True
    if hasattr(mat, "surface_render_method"):
        if mat.surface_render_method == 'OPAQUE':
            mat.surface_render_method = 'HASHED'
    elif hasattr(mat, "blend_method"):
        if mat.blend_method == 'OPAQUE':
            mat.blend_method = 'HASHED'
    nodes = mat.node_tree.nodes
    links = mat.node_tree.links
    nodes.clear()
    emission = nodes.new("ShaderNodeEmission")
    emission.name = "Toon_OutlineEmission"
    emission.label = "Toon_OutlineEmission"
    emission.inputs["Color"].default_value = (*props.outline_color, 1.0)
    emission.inputs[1].default_value = 1.0
    emission.location = (-300, 100)
    transparent = nodes.new("ShaderNodeBsdfTransparent")
    transparent.location = (-300, -100)
    light_path = nodes.new("ShaderNodeLightPath")
    light_path.location = (-600, 200)
    output = nodes.new("ShaderNodeOutputMaterial")
    output.location = (250, 0)
    if _is_cycles():
        backface = nodes.new("ShaderNodeNewGeometry")
        backface.label = "Toon_Backfacing"
        backface.location = (-600, -100)
        mix_backface = nodes.new("ShaderNodeMixShader")
        mix_backface.location = (-100, -50)
        mix_shadow = nodes.new("ShaderNodeMixShader")
        mix_shadow.location = (80, 0)
        links.new(backface.outputs["Backfacing"], mix_backface.inputs[0])
        links.new(emission.outputs["Emission"], mix_backface.inputs[1])
        links.new(transparent.outputs["BSDF"], mix_backface.inputs[2])
        links.new(light_path.outputs["Is Shadow Ray"], mix_shadow.inputs[0])
        links.new(mix_backface.outputs["Shader"], mix_shadow.inputs[1])
        links.new(transparent.outputs["BSDF"], mix_shadow.inputs[2])
        links.new(mix_shadow.outputs["Shader"], output.inputs["Surface"])
    else:
        mix = nodes.new("ShaderNodeMixShader")
        mix.location = (0, 0)
        links.new(light_path.outputs["Is Shadow Ray"], mix.inputs[0])
        links.new(emission.outputs["Emission"], mix.inputs[1])
        links.new(transparent.outputs["BSDF"], mix.inputs[2])
        links.new(mix.outputs["Shader"], output.inputs["Surface"])
    return mat