"""レンダーエンジン判定とエンジン別最適化設定"""
import bpy
def is_eevee(context=None):
    """
    現在のレンダーエンジンがEevee/Eevee Nextかどうかを判定
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        bool: Eevee系レンダーエンジンの場合True
    """
    if context is None:
        context = bpy.context
    scene = context.scene
    return scene.render.engine in ('BLENDER_EEVEE', 'BLENDER_EEVEE_NEXT')
def is_cycles(context=None):
    """
    現在のレンダーエンジンがCyclesかどうかを判定
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        bool: Cyclesレンダーエンジンの場合True
    """
    if context is None:
        context = bpy.context
    scene = context.scene
    return scene.render.engine == 'CYCLES'
def get_engine_name(context=None):
    """
    現在のレンダーエンジン名を取得
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        str: レンダーエンジン名（'EEVEE', 'CYCLES', 'OTHER'）
    """
    if context is None:
        context = bpy.context
    scene = context.scene
    engine = scene.render.engine
    if engine in ('BLENDER_EEVEE', 'BLENDER_EEVEE_NEXT'):
        return 'EEVEE'
    elif engine == 'CYCLES':
        return 'CYCLES'
    else:
        return 'OTHER'
def apply_eevee_optimizations(scene):
    """
    Eevee用の推奨レンダー設定を適用
    Args:
        scene: Blenderシーンオブジェクト
    Returns:
        list: 適用した設定のリスト
    """
    settings_applied = []
    if hasattr(scene.eevee, 'use_soft_shadows'):
        scene.eevee.use_soft_shadows = False
        settings_applied.append("Soft Shadows OFF")
    if hasattr(scene.eevee, 'use_gtao'):
        scene.eevee.use_gtao = False
        settings_applied.append("Ambient Occlusion OFF")
    if hasattr(scene.eevee, 'use_bloom'):
        scene.eevee.use_bloom = False
        settings_applied.append("Bloom OFF")
    if hasattr(scene.eevee, 'shadow_cube_size'):
        scene.eevee.shadow_cube_size = '512'
        settings_applied.append("Shadow Cube Size: 512")
    if hasattr(scene.eevee, 'shadow_cascade_size'):
        scene.eevee.shadow_cascade_size = '512'
        settings_applied.append("Shadow Cascade Size: 512")
    return settings_applied
def apply_cycles_optimizations(scene):
    """
    Cycles用の推奨レンダー設定を適用
    Args:
        scene: Blenderシーンオブジェクト
    Returns:
        list: 適用した設定のリスト
    """
    settings_applied = []
    if hasattr(scene.cycles, 'samples'):
        if scene.cycles.preview_samples > 128:
            scene.cycles.preview_samples = 128
            settings_applied.append("Preview Samples: 128")
    if hasattr(scene.cycles, 'use_denoising'):
        if scene.cycles.use_denoising:
            scene.cycles.use_denoising = False
            settings_applied.append("Denoising: OFF (not needed for toon shading)")
    return settings_applied
def apply_render_optimizations(context=None):
    """
    現在のレンダーエンジンに応じた最適化設定を自動適用
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        tuple: (engine_name, settings_applied)
    """
    if context is None:
        context = bpy.context
    scene = context.scene
    engine_name = get_engine_name(context)
    settings_applied = []
    if scene.view_settings.view_transform != 'Standard':
        old_transform = scene.view_settings.view_transform
        scene.view_settings.view_transform = 'Standard'
        settings_applied.append(f"Color Management: {old_transform} -> Standard")
    if engine_name == 'EEVEE':
        eevee_settings = apply_eevee_optimizations(scene)
        settings_applied.extend(eevee_settings)
    elif engine_name == 'CYCLES':
        cycles_settings = apply_cycles_optimizations(scene)
        settings_applied.extend(cycles_settings)
    return (engine_name, settings_applied)
def get_recommended_shading_mode(context=None):
    """
    現在のレンダーエンジンに推奨されるシェーディングモードを取得
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        str: 'LIGHTING' または 'VECTOR'
    """
    engine_name = get_engine_name(context)
    if engine_name == 'EEVEE':
        return 'LIGHTING'
    elif engine_name == 'CYCLES':
        return 'VECTOR'
    else:
        return 'VECTOR'
def get_engine_display_name(context=None):
    """
    UIに表示するレンダーエンジン名を取得
    Args:
        context: Blenderコンテキスト（Noneの場合はbpy.contextを使用）
    Returns:
        str: 表示用のレンダーエンジン名
    """
    if context is None:
        context = bpy.context
    scene = context.scene
    engine = scene.render.engine
    engine_names = {
        'BLENDER_EEVEE': 'Eevee',
        'BLENDER_EEVEE_NEXT': 'Eevee Next',
        'CYCLES': 'Cycles',
        'BLENDER_WORKBENCH': 'Workbench',
    }
    return engine_names.get(engine, engine)