"""コア - マテリアル設定の読み取り (逆同期)"""
import bpy
def get_toon_settings_from_object(obj):
    """オブジェクトのマテリアル/モディファイアからトゥーン設定を読み取る"""
    data = {}
    print(f"[OMOSEN+] Scanning object '{obj.name}' (Type: {obj.type})")
    if obj.type == 'MESH':
        outline_mod = None
        for mod in obj.modifiers:
            if mod.type == 'SOLIDIFY' and (mod.name.startswith("ToonOutline") or mod.use_flip_normals):
                data["outline_width"] = mod.thickness
                if mod.vertex_group:
                    data["outline_vertex_group"] = mod.vertex_group
                outline_mod = mod
                print(f"[OMOSEN+] -> Outline Modifier found (Thickness: {mod.thickness}, Offset: {mod.material_offset})")
                break
        active_idx = obj.active_material_index
        slots_to_check = list(range(len(obj.material_slots)))
        outline_slot_idx = -1
        if outline_mod:
            for i, slot in enumerate(obj.material_slots):
                mat = slot.material
                if mat and mat.name.startswith("OutlineMaterial"):
                    outline_slot_idx = i
                    print(f"[OMOSEN+] -> Outline slot found by name: {outline_slot_idx+1}")
                    break
            if outline_slot_idx == -1:
                idx = active_idx + outline_mod.material_offset
                if 0 <= idx < len(obj.material_slots):
                    outline_slot_idx = idx
                    print(f"[OMOSEN+] -> Outline slot calculated: {outline_slot_idx+1}")
        order = [active_idx]
        if outline_slot_idx != -1 and outline_slot_idx not in order:
            order.append(outline_slot_idx)
        for i in slots_to_check:
            if i not in order:
                order.append(i)
        for i in order:
            slot = obj.material_slots[i]
            mat = slot.material
            if not mat:
                continue
            if not mat.use_nodes:
                continue
            nodes = mat.node_tree.nodes
            print(f"[OMOSEN+] -> Checking Slot {i+1}: '{mat.name}' ({len(nodes)} nodes)")
            for node in nodes:
                label_str = node.label if node.label else ""
                name_str = node.name if node.name else ""
                full_id = (label_str + name_str).lower()
                ntype = node.type
                is_mix_type = ntype in {'MIX_RGB', 'MIX'}
                is_mix_node = (ntype == 'MIX')
                if "toon_lightboost" in full_id and ntype == 'MATH' and "light_boost" not in data:
                    data["light_boost"] = node.inputs[1].default_value
                elif "toon_hardness" in full_id and "shadow_hardness" not in data:
                    if ntype == 'MAP_RANGE':
                        v1 = node.inputs[1].default_value
                        v2 = node.inputs[2].default_value
                        diff = v2 - v1
                        if diff > 0.000001:
                            data["shadow_hardness"] = min(2.0 / diff, 1000.0)
                        data["shadow_step"] = (v1 + v2) / 2.0
                elif "toon_mix" in full_id and is_mix_type and "shadow_strength" not in data:
                    data["shadow_strength"] = node.inputs[0].default_value
                elif "toon_addspec" in full_id and is_mix_type and "use_specular" not in data:
                    data["use_specular"] = node.inputs[0].default_value > 0.5
                elif "toon_specmix" in full_id and is_mix_type and "specular_color" not in data:
                    if is_mix_node:
                        soc = node.inputs.get("B") or node.inputs.get("Color2") or (node.inputs[7] if len(node.inputs) > 7 else None)
                        if soc:
                            col = soc.default_value
                            data["specular_color"] = (col[0], col[1], col[2])
                    else:
                        col = node.inputs[2].default_value
                        data["specular_color"] = (col[0], col[1], col[2])
                elif "toon_specramp" in full_id and ntype == 'VALTORGB' and "specular_size" not in data:
                    data["specular_size"] = 1.0 - node.color_ramp.elements[0].position
                elif "toon_colorramp" in full_id and ntype == 'VALTORGB' and "interpolation" not in data:
                    data["interpolation"] = node.color_ramp.interpolation
                    cnt = len(node.color_ramp.elements)
                    data["shadow_levels"] = 3 if cnt >= 3 else 2
                    if cnt >= 2:
                       c0 = node.color_ramp.elements[0].color
                       if cnt >= 3:
                           data["shadow_color_2"] = (c0[0], c0[1], c0[2])
                           c1 = node.color_ramp.elements[1].color
                           data["shadow_color"] = (c1[0], c1[1], c1[2])
                           c2 = node.color_ramp.elements[2].color
                           data["highlight_color"] = (c2[0], c2[1], c2[2])
                       else:
                           data["shadow_color"] = (c0[0], c0[1], c0[2])
                           c1 = node.color_ramp.elements[1].color
                           data["highlight_color"] = (c1[0], c1[1], c1[2])
                elif "toon_rimmap" in full_id and ntype == 'MAP_RANGE' and "rim_light_width" not in data:
                     data["rim_light_width"] = 1.0 - node.inputs[1].default_value
                elif "toon_rimmix" in full_id and is_mix_type and "rim_light_color" not in data:
                    if is_mix_node:
                        soc = node.inputs.get("B") or node.inputs.get("Color2") or (node.inputs[7] if len(node.inputs) > 7 else None)
                        if soc:
                            col = soc.default_value
                            data["rim_light_color"] = (col[0], col[1], col[2])
                    else:
                        col = node.inputs[2].default_value
                        data["rim_light_color"] = (col[0], col[1], col[2])
                elif "toon_addrim" in full_id and is_mix_type and "use_rim_light" not in data:
                    val = node.inputs[0].default_value
                    data["use_rim_light"] = val > 0.001
                    data["rim_light_strength"] = val
                elif "toon_shadingmix" in full_id and is_mix_type and "shading_method" not in data:
                    data["shading_method"] = 'VECTOR' if node.inputs[0].default_value > 0.5 else 'LIGHTING'
                elif "toon_lightvector" in full_id and ntype == 'RGB' and "light_direction" not in data:
                    col = node.outputs[0].default_value
                    data["light_direction"] = (col[0], col[1], col[2])
                elif "toon_outlineemission" in full_id and ntype == 'EMISSION' and "outline_color" not in data:
                    col = node.inputs["Color"].default_value
                    data["outline_color"] = (col[0], col[1], col[2])
        if outline_mod:
            data["material_offset"] = outline_mod.material_offset
    elif obj.type == 'GPENCIL':
        if obj.name.startswith("ToonDrawing") or obj.name.startswith("ToonEdgeLines"):
            for mat in obj.data.materials:
                if hasattr(mat, 'grease_pencil_settings'):
                    col = mat.grease_pencil_settings.color
                    data["edge_line_color"] = (col[0], col[1], col[2])
                    for mod in obj.modifiers:
                        if mod.type == 'GP_THICKNESS':
                            data["edge_line_thickness"] = mod.thickness_factor
                            break
                    break
    return data
def sync_props_from_active_object(scene):
    """アクティブオブジェクトからプロパティを同期"""
    if not scene:
        print("[OMOSEN+] Error: No scene provided to sync.")
        return
    obj = bpy.context.active_object
    if not obj:
        print("[OMOSEN+] No active object to sync.")
        return
    print(f"[OMOSEN+] Starting sync from '{obj.name}'...")
    data = get_toon_settings_from_object(obj)
    if not data:
        print(f"[OMOSEN+] No toon settings found in '{obj.name}'")
        return
    props = scene.omosen_props
    print(f"[OMOSEN+] Setting is_syncing = True")
    props.is_syncing = True
    print(f"[OMOSEN+] Data keys found: {list(data.keys())}")
    try:
        if "outline_width" in data: props.outline_width = data["outline_width"]
        if "outline_color" in data: props.outline_color = data["outline_color"]
        if "outline_vertex_group" in data: props.outline_vertex_group = data["outline_vertex_group"]
        if "material_offset" in data: props.material_offset = data["material_offset"]
        if "light_boost" in data: props.light_boost = data["light_boost"]
        if "shadow_hardness" in data: props.shadow_hardness = data["shadow_hardness"]
        if "shadow_step" in data: props.shadow_step = data["shadow_step"]
        if "shadow_strength" in data: props.shadow_strength = data["shadow_strength"]
        if "use_specular" in data: props.use_specular = data["use_specular"]
        if "specular_color" in data: props.specular_color = data["specular_color"]
        if "specular_size" in data: props.specular_size = data["specular_size"]
        if "interpolation" in data: props.interpolation = data["interpolation"]
        if "shadow_levels" in data: props.shadow_levels = data["shadow_levels"]
        if "shadow_color" in data: props.shadow_color = data["shadow_color"]
        if "shadow_color_2" in data: props.shadow_color_2 = data["shadow_color_2"]
        if "highlight_color" in data: props.highlight_color = data["highlight_color"]
        if "rim_light_width" in data: props.rim_light_width = data["rim_light_width"]
        if "rim_light_color" in data: props.rim_light_color = data["rim_light_color"]
        if "use_rim_light" in data: props.use_rim_light = data["use_rim_light"]
        if "rim_light_strength" in data: props.rim_light_strength = data["rim_light_strength"]
        if "shading_method" in data: props.shading_method = data["shading_method"]
        if "light_direction" in data: props.light_direction = data["light_direction"]
        if "edge_line_color" in data: props.edge_line_color = data["edge_line_color"]
        if "edge_line_thickness" in data: props.edge_line_thickness = data["edge_line_thickness"]
        print(f"[OMOSEN+] Sync completed successfully!")
    except Exception as e:
        import traceback
        print(f"[OMOSEN+] Sync Error: {e}")
        print(traceback.format_exc())
    finally:
        props.is_syncing = False
        print(f"[OMOSEN+] Setting is_syncing = False")