"""オペレータ - 設定の入出力"""
import bpy
from bpy.props import StringProperty
from bpy_extras.io_utils import ImportHelper, ExportHelper
from bpy.types import Operator
from ..core.io import (
    save_settings_to_file, 
    load_settings_from_file, 
    add_and_save_preset,
    remove_and_save_preset,
    load_custom_presets, 
    apply_custom_preset_by_name
)
class OMOSEN_OT_ExportSettings(Operator, ExportHelper):
    """設定を保存"""
    bl_idname = "omosen.export_settings"
    bl_label = "Export Settings"
    bl_description = "Export current settings as a JSON file"
    filename_ext = ".json"
    filter_glob: StringProperty(
        default="*.json",
        options={'HIDDEN'},
        maxlen=255,
    )
    def execute(self, context):
        if save_settings_to_file(context, self.filepath):
            self.report({'INFO'}, bpy.app.translations.pgettext_iface("Settings saved: {}").format(self.filepath))
            return {'FINISHED'}
        else:
            self.report({'ERROR'}, bpy.app.translations.pgettext_iface("Failed to save settings"))
            return {'CANCELLED'}
class OMOSEN_OT_ImportSettings(Operator, ImportHelper):
    """設定を読み込み"""
    bl_idname = "omosen.import_settings"
    bl_label = "Import Settings"
    bl_description = "Import settings from a JSON file"
    filename_ext = ".json"
    filter_glob: StringProperty(
        default="*.json",
        options={'HIDDEN'},
        maxlen=255,
    )
    def execute(self, context):
        if load_settings_from_file(context, self.filepath):
            self.report({'INFO'}, bpy.app.translations.pgettext_iface("Settings loaded: {}").format(self.filepath))
            return {'FINISHED'}
        else:
            self.report({'ERROR'}, bpy.app.translations.pgettext_iface("Failed to load settings"))
            return {'CANCELLED'}
class OMOSEN_OT_AddCustomPreset(Operator):
    """現在の設定をプリセットとして保存"""
    bl_idname = "omosen.add_custom_preset"
    bl_label = "Save Current Settings"
    bl_description = "Save all current parameters as a new preset"
    bl_options = {'REGISTER', 'UNDO'}
    preset_name: StringProperty(name="Preset Name", default="My Preset")
    def execute(self, context):
        add_and_save_preset(context, self.preset_name)
        self.report({'INFO'}, bpy.app.translations.pgettext_iface("Preset '{}' saved").format(self.preset_name))
        return {'FINISHED'}
    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)
class OMOSEN_OT_RemoveCustomPreset(Operator):
    """プリセットを削除"""
    bl_idname = "omosen.remove_custom_preset"
    bl_label = "Remove Preset"
    bl_description = "Remove the selected preset"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        props = context.scene.omosen_props
        index = props.active_preset_index
        if 0 <= index < len(props.custom_presets):
            name = props.custom_presets[index].name
            remove_and_save_preset(context, name)
            props.active_preset_index = max(0, index - 1)
            self.report({'INFO'}, bpy.app.translations.pgettext_iface("Preset '{}' removed").format(name))
        return {'FINISHED'}
class OMOSEN_OT_ApplyCustomPreset(Operator):
    """プリセットを適用"""
    bl_idname = "omosen.apply_custom_preset"
    bl_label = "Apply Preset"
    bl_description = "Apply the selected preset to the scene"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        props = context.scene.omosen_props
        index = props.active_preset_index
        if 0 <= index < len(props.custom_presets):
            name = props.custom_presets[index].name
            if apply_custom_preset_by_name(context, name):
                self.report({'INFO'}, bpy.app.translations.pgettext_iface("Preset '{}' applied").format(name))
                from ..core import materials
                materials.update_materials(context)
                return {'FINISHED'}
        self.report({'WARNING'}, bpy.app.translations.pgettext_iface("No preset found to apply"))
        return {'CANCELLED'}