"""オペレータ - ユーティリティ（便利機能）"""
import bpy
from bpy.types import Operator
class OMOSEN_OT_CreateDrawingGPencil(Operator):
    """描き込み用グリースペンシルを作成"""
    bl_idname = "omosen.create_drawing_gpencil"
    bl_label = "Create Drawing Pen"
    bl_description = "Create a grease pencil object for drawing on mesh surface"
    bl_options = {'REGISTER', 'UNDO'}
    def execute(self, context):
        active_obj = context.active_object
        if not active_obj or active_obj.type != 'MESH':
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("Please select a mesh object"))
            return {'CANCELLED'}
        gp_data = bpy.data.grease_pencils.new(f"ToonDrawing_{active_obj.name}")
        if hasattr(gp_data, "layers"):
            layer = gp_data.layers.get("Lines") or gp_data.layers.new(name="Lines")
            if hasattr(layer, "frames") and not layer.frames:
                layer.frames.new(context.scene.frame_current)
        gp_obj = bpy.data.objects.new(f"ToonDrawing_{active_obj.name}", gp_data)
        context.collection.objects.link(gp_obj)
        gp_obj.parent = active_obj
        gp_obj.matrix_world = active_obj.matrix_world
        props = context.scene.omosen_props
        mat = bpy.data.materials.new(f"ToonDrawingMat_{active_obj.name}")
        bpy.data.materials.create_gpencil_data(mat)
        mat.grease_pencil.color = (*props.edge_line_color, 1.0)
        gp_data.materials.append(mat)
        gp_obj.data.stroke_depth_order = '3D'
        gp_obj.show_in_front = True
        bpy.ops.object.select_all(action='DESELECT')
        gp_obj.select_set(True)
        context.view_layer.objects.active = gp_obj
        context.view_layer.update()
        is_gpv3 = bpy.app.version >= (4, 3, 0)
        gp_paint_mode = 'PAINT_GREASE_PENCIL' if is_gpv3 else 'PAINT_GPENCIL'
        try:
            bpy.ops.object.mode_set(mode=gp_paint_mode)
        except Exception as e:
            print(f"OMOSEN+ Mode Switch Error: {e}")
            self.report({'WARNING'}, bpy.app.translations.pgettext_iface("Failed to switch mode. Please switch manually."))
        ts = context.tool_settings
        if is_gpv3 and hasattr(ts, "grease_pencil"):
            ts.grease_pencil.stroke_placement = 'SURFACE'
        elif hasattr(ts, "gpencil_stroke_placement_view3d"):
            ts.gpencil_stroke_placement_view3d = 'SURFACE'
        settings = None
        if is_gpv3 and hasattr(ts, "grease_pencil_paint"):
            settings = ts.grease_pencil_paint
        elif hasattr(ts, "gpencil_paint"):
            settings = ts.gpencil_paint
        if settings:
            try:
                settings.brush = bpy.data.brushes.get("Pencil") or bpy.data.brushes.new("Pencil", mode=gp_paint_mode)
            except:
                print("OMOSEN+ Info: Could not set active brush automatically (GPv3)")
            brush = getattr(settings, "brush", None)
            if brush:
                if hasattr(brush, "gpencil_settings"):
                    brush.gpencil_settings.pen_strength = 1.0
                if hasattr(brush, "size"):
                    brush.size = 20
        self.report({'INFO'}, bpy.app.translations.pgettext_iface("Drawing mode started (Esc to end)"))
        return {'FINISHED'}
class OMOSEN_OT_ToggleLineArt(Operator):
    """Line Art（ビューポート表示）の切り替え"""
    bl_idname = "omosen.toggle_line_art"
    bl_label = "Toggle Line Art"
    bl_description = "Toggle real-time edge line display in viewport"
    def execute(self, context):
        props = context.scene.omosen_props
        props.use_line_art = not props.use_line_art
        active_obj = context.active_object
        if not active_obj or active_obj.type != 'MESH':
            return {'FINISHED'}
        gp_name = f"ToonEdgeLines_{active_obj.name}"
        if props.use_line_art:
            if gp_name not in bpy.data.objects:
                gp_data = bpy.data.grease_pencils.new(gp_name)
                if hasattr(gp_data, "layers"):
                    layer = gp_data.layers.get("Lines") or gp_data.layers.new(name="Lines")
                    if hasattr(layer, "frames") and not layer.frames:
                        layer.frames.new(context.scene.frame_current)
                gp_obj = bpy.data.objects.new(gp_name, gp_data)
                context.collection.objects.link(gp_obj)
                is_gpv3 = bpy.app.version >= (4, 3, 0)
                if is_gpv3:
                    mod = gp_obj.modifiers.new(name="Line Art", type='LINEART')
                else:
                    mod = gp_obj.grease_pencil_modifiers.new(name="Line Art", type='GP_LINEART')
                mod.source_type = 'OBJECT'
                mod.source_object = active_obj
                mod.target_layer = "Lines"
                if hasattr(mod, "thickness"):
                    mod.thickness = props.edge_line_thickness
                elif hasattr(mod, "line_thickness"):
                    mod.line_thickness = props.edge_line_thickness
                elif hasattr(mod, "stroke_thickness"):
                    mod.stroke_thickness = props.edge_line_thickness
                mat = bpy.data.materials.new(f"ToonEdgeMat_{active_obj.name}")
                bpy.data.materials.create_gpencil_data(mat)
                mat.grease_pencil.color = (*props.edge_line_color, 1.0)
                gp_data.materials.append(mat)
        else:
            if gp_name in bpy.data.objects:
                obj = bpy.data.objects[gp_name]
                bpy.data.objects.remove(obj)
        return {'FINISHED'}
class OMOSEN_OT_ToggleFreestyle(Operator):
    """Freestyle（レンダリング）の切り替え"""
    bl_idname = "omosen.toggle_freestyle"
    bl_label = "Toggle Freestyle"
    bl_description = "Toggle high-quality line rendering (Freestyle)"
    def execute(self, context):
        props = context.scene.omosen_props
        props.use_freestyle = not props.use_freestyle
        context.scene.render.use_freestyle = props.use_freestyle
        if props.use_freestyle:
            view_layer = context.view_layer
            if not view_layer.freestyle_settings.linesets:
                lineset = view_layer.freestyle_settings.linesets.new("ToonLines")
                lineset.select_crease = True
                lineset.select_border = True
                lineset.select_contour = True
        return {'FINISHED'}